<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LegalCase;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Models\ActivityLog;

class CaseController extends Controller
{
    // 1. Show All Cases
    public function index()
    {
        $cases = LegalCase::with(['client', 'lawyer'])
            ->orderByRaw('CASE WHEN court_date IS NULL THEN 1 ELSE 0 END, court_date ASC')
            ->paginate(10);

        return view('admin.cases.index', compact('cases'));
    }

    // 2. Show the "Create Case" Form
    public function create()
    {
        // We need lists of Clients and Lawyers for the dropdowns
        $clients = User::role('client')->get();
        $lawyers = User::role(['lawyer', 'super_admin'])->get();

        return view('admin.cases.create', compact('clients', 'lawyers'));
    }

    // 3. Save the new Case
    public function store(Request $request)
    {
        // 1. VALIDATION
        $validated = $request->validate([
            // Core Info
            'title' => 'required|string|max:255',
            'case_number' => 'required|string|max:50|unique:legal_cases,case_number',
            'description' => 'nullable|string',
            'case_stage' => 'required|string',

            // People
            'client_id' => 'required|exists:users,id',
            'user_id' => 'required|exists:users,id', // <--- Input name is 'user_id'

            // Court Details
            'court_name' => 'nullable|string|max:255',
            'judge_name' => 'nullable|string|max:255',
            'court_date' => 'nullable|date',
            'opposing_counsel' => 'nullable|string|max:255',

            // Police Info
            'police_station' => 'nullable|string|max:255',
            'police_ob_number' => 'nullable|string|max:50',
        ]);

        // 2. PREPARE DATA
        $validated['status'] = 'open';

        // --- THE FIX ---
        // Map the form input 'user_id' to the database column 'assigned_lawyer_id'
        $validated['assigned_lawyer_id'] = $validated['user_id'];

        // Remove 'user_id' so Eloquent doesn't try to insert a column that doesn't exist
        unset($validated['user_id']);

        // 3. CREATE RECORD
        $case = LegalCase::create($validated);

        // 4. LOG ACTIVITY
        if (class_exists('App\Models\ActivityLog')) {
            \App\Models\ActivityLog::record('Create', 'Case', "Opened new case file: {$case->case_number}");
        }

        return redirect()->route('admin.cases.index')->with('success', 'New Matter opened successfully!');
    }

    public function show($id)
    {
        // Fetch the case with its relationships
        // $case = LegalCase::with(['client', 'lawyer'])->findOrFail($id);
        $case = LegalCase::with(['client', 'lawyer', 'documents', 'timeline', 'invoices'])->findOrFail($id);

        return view('admin.cases.show', compact('case'));
    }

    // 4. Show the Edit Form
    public function edit($id)
    {
        $case = LegalCase::findOrFail($id);
        $clients = User::role('client')->get();
        $lawyers = User::role(['lawyer', 'super_admin'])->get();

        return view('admin.cases.edit', compact('case', 'clients', 'lawyers'));
    }

    // 5. Update the Case Logic
    public function update(Request $request, $id)
    {
        $case = LegalCase::findOrFail($id);

        $validated = $request->validate([
            // 1. Core Info
            'title' => 'required|string|max:255',
            'case_number' => 'required|string|max:50|unique:legal_cases,case_number,' . $case->id, // Unique but ignore current case
            'description' => 'nullable|string',
            'status' => 'required|in:open,hearing,pending,closed,archived',
            'case_stage' => 'nullable|string', // Pre-Trial, Mention, etc.

            // 2. People
            'client_id' => 'required|exists:users,id',
            'user_id' => 'required|exists:users,id', // This is the Assigned Advocate

            // 3. Court Details
            'court_name' => 'nullable|string|max:255',
            'judge_name' => 'nullable|string|max:255',
            'court_date' => 'nullable|date',
            'opposing_counsel' => 'nullable|string|max:255',

            // 4. Police Details
            'police_station' => 'nullable|string|max:255',
            'police_ob_number' => 'nullable|string|max:50',
        ]);

        // Mass assignment update (Safe because we validated everything)
        $case->update($validated);

        return redirect()
            ->route('admin.cases.show', $case->id)
            ->with('success', 'Case file updated successfully.');
    }

    // 6. Delete (Soft Delete)
    public function destroy($id)
    {
        $case = LegalCase::findOrFail($id);
        $case->delete(); // Soft delete (can be restored if needed)

        return redirect()->route('admin.dashboard')->with('success', 'Case archived successfully.');
    }
}
