<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

class ClientController extends Controller
{
    /**
     * 1. List all Clients
     */
    public function index()
    {
        $clients = User::role('client')
            ->withCount('cases')
            ->latest()
            ->paginate(10);

        return view('admin.clients.index', compact('clients'));
    }

    /**
     * 2. Show Create Form
     */
    public function create()
    {
        return view('admin.clients.create');
    }

    /**
     * 3. Store New Client
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|max:20',
            'address' => 'nullable|string|max:500',
            'kra_pin' => 'nullable|string|max:20', // Add this line
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'password' => Hash::make('Password123!'), // Default password
            'kra_pin' => $request->kra_pin, // Add this line
        ]);

        $user->assignRole('client');

        return redirect()->route('admin.clients.show', $user->id)
            ->with('success', 'Client account created successfully.');
    }

    /**
     * 4. Show Client Profile (Stats & Cases)
     */
    public function show($id)
    {
        $client = User::with(['cases', 'cases.invoices'])
            ->role('client')
            ->findOrFail($id);

        // 1. KPI Stats
        $totalCases = $client->cases->count();
        $activeCases = $client->cases->where('status', '!=', 'closed')->count();
        $totalBilled = $client->cases->flatMap->invoices->sum('total_amount');
        $totalUnpaid = $client->cases->flatMap->invoices->where('status', '!=', 'paid')->sum('total_amount');

        // 2. CHART DATA: Client's Billing History (Last 6 Months)
        // We look at invoices belonging to cases belonging to this user
        $billingData = \App\Models\Invoice::whereHas('case', function ($q) use ($client) {
            $q->where('client_id', $client->id);
        })
            ->selectRaw("DATE_FORMAT(created_at,'%M') as month, sum(total_amount) as total")
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderByRaw("MIN(created_at) ASC")
            ->get();

        $chartLabels = $billingData->pluck('month');
        $chartValues = $billingData->pluck('total');

        return view('admin.clients.show', compact(
            'client',
            'totalCases',
            'activeCases',
            'totalBilled',
            'totalUnpaid',
            'chartLabels',
            'chartValues'
        ));
    }

    /**
     * 5. Show Edit Form
     */
    public function edit($id)
    {
        $client = User::role('client')->findOrFail($id);
        return view('admin.clients.edit', compact('client'));
    }

    /**
     * 6. Update Client
     */
    public function update(Request $request, $id)
    {
        $client = User::role('client')->findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $client->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'kra_pin' => 'nullable|string|max:20', // Add this line
        ]);

        $client->update([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'kra_pin' => $request->kra_pin, // Add this line
        ]);

        return redirect()->route('admin.clients.show', $client->id)
            ->with('success', 'Client profile updated.');
    }

    /**
     * 7. Delete Client
     */
    public function destroy(string $id)
    {
        $client = User::role('client')->findOrFail($id);

        // Optional: Check if they have active cases and prevent delete?
        // For now, we allow delete (cascading logic in DB handles cases usually, or they become orphaned)

        $client->delete();

        return redirect()->route('admin.clients.index')
            ->with('success', 'Client account and access removed.');
    }
}
